#ifndef TMYZEE_HH
#define TMYZEE_HH

#if !defined (__CINT__) || defined (__MAKECINT__)

#include "TH1.h"
#include "TH2.h"
#include "TProfile.h"
#include <vector>
#include "TLorentzVector.h"
#include "TVector3.h"
#include <Stntuple/loop/TStnModule.hh>
#include <Stntuple/obj/TStnHeaderBlock.hh>
#include <Stntuple/obj/TStnEvent.hh>
#include <Stntuple/obj/TStnElectronBlock.hh>
#include <Stntuple/obj/TStnPhotonBlock.hh>
#include <Stntuple/obj/TStnTrackBlock.hh>
#include <Stntuple/obj/TPhoenixElectronBlock.hh> // need for phoenix electrons
#include <Stntuple/photon/TPhotonUtil.hh>
#include <Stntuple/alg/TStntuple.hh>
#include <Stntuple/data/TStnBeamPos.hh>
#include <Stntuple/data/TStnBeamPosBlock.hh>
#include <Stntuple/obj/TStnDBManager.hh>

#endif

class TMyVertexFilterModule;
 
class TMyZeeFilterModule: public TStnModule {
public:

  struct ElectronGeneral_t {
    
    // Note, the electron energy in these histos is the EM energy only
    // unless otherwise noted
  
    //_______________________before cuts
    
    TH1F* fEleNcand_b;                  // number of Electron candidates
    TH1F* fEleNtower_b;                 // number of towers in the cluster
    TH1F* fEleDetector_b;               // 0: CEM, 1: PEM, 2: Forward
    TH1F* fEleEt_raw_b;                 // raw Et(EM+HAD) with respect to vertex and CES
    TH1F* fEleEt_corr_b;                // Et correction (for tower response)
    TH1F* fEleEt_b;                     // corrected Et(EM) with respect to vertex and CES
    TH1F* fEleE_em_b;                   // corrected EM Energy
    TH1F* fEleE_total_b;                // total (EM+HAD) corrected Energy
    TH1F* fEleEta_det_b;                // detector Eta
    TH1F* fEleEta_b;                    // event Eta 
    TH1F* fElePhi_b;                    // Phi 
    TH1F* fEleTheta_b;                  // Theta 
    TH1F* fEleHadEm_b;                  // Had/Em 
    TH1F* fEleCalIso4_raw_b;            // raw cal Iso in cone R=0.4 (sum Et in R=0.4)
    TH1F* fEleCalIso4_corr_b;           // cal Iso in cone R=0.4 corrected for leakage & multiple vert.
    TH1F* fEleCalIso4Fr_raw_b;          // raw CalIso4/EtEle
    TH1F* fEleCalIso4Fr_corr_b;         // corr. CalIso4/EtEle

    //_______________________after cuts
    
    TH1F* fEleNcand_a;                  
    TH1F* fEleNtower_a;                 
    TH1F* fEleDetector_a;               
    TH1F* fEleEt_raw_a;                 
    TH1F* fEleEt_corr_a;                
    TH1F* fEleEt_a;                    
    TH1F* fEleE_em_a;                  
    TH1F* fEleE_total_a;                
    TH1F* fEleEta_det_a;                
    TH1F* fEleEta_a;                    
    TH1F* fElePhi_a;                    
    TH1F* fEleTheta_a;                  
    TH1F* fEleHadEm_a;                  
    TH1F* fEleCalIso4_raw_a;            
    TH1F* fEleCalIso4_corr_a;           
    TH1F* fEleCalIso4Fr_raw_a;          
    TH1F* fEleCalIso4Fr_corr_a;         

  };

  //----------- histos for two leading central electrons
  struct CEMElectron_t {

    // Note, the electron energy in these histos is the EM energy only
    // unless otherwise noted
    
    //_______________________before cuts

    TH1F* fCemNcand_b;                  // number of CEM candidates
    TH1F* fCemNtower_b[2];                 // number of towers in the cluster
    TH1F* fCemEt_raw_b[2];                 // raw Et(EM+HAD) with respect to vertex and CES
    TH1F* fCemEt_corr_b[2];                // Et correction (for tower response)
    TH1F* fCemEt_b[2];                     // corrected Et(EM) with respect to vertex and CES
    TH1F* fCemE_em_b[2];                   // corrected EM Energy
    TH1F* fCemE_total_b[2];                // total (EM+HAD) corrected Energy
    TH1F* fCemEta_det_b[2];                // detector Eta
    TH1F* fCemEta_b[2];                    // event Eta 
    TH1F* fCemPhi_b[2];                    // Phi 
    TH1F* fCemTheta_b[2];                  // Theta 
    TH1F* fCemHadEm_b[2];                  // Had/Em 
    TH1F* fCemLshr_b[2];                   // Lshr 
    TH1F* fCemCalIso4_raw_b[2];            // raw cal Iso in cone R=0.4 (sum Et in R=0.4)
    TH1F* fCemCalIso4_corr_b[2];           // cal Iso in cone R=0.4 corrected for leakage & multiple vert.
    TH1F* fCemCalIso4Fr_raw_b[2];          // raw CalIso4/EtEle
    TH1F* fCemCalIso4Fr_corr_b[2];         // corr. CalIso4/EtEle
    TH1F* fCemChi2strip_b[2];              // CES strip Chi2
    TH1F* fCemChi2_b[2];                   // CES (strip+wire)/2 Chi2
    TH1F* fCemFid_b[2];                // Fiduciality: "CES-based"=1, "Trk-based"=4 (for central only?)
    TH1F* fCemFidCes_b[2];             // CES Fiduciality code (for central only?)
    TH1F* fCemFidTrk_b[2];             // Trk Fiduciality code (for central only?)
    TH1F* fCemEleXces_b[2];            // CES x position of EM cluster
    TH1F* fCemTrkXces_b[2];            // CES x position of a track   
    TH1F* fCemConvStat_b[2];               // Conversion status
    TH1F* fCemEP_b[2];                     // E(cal cluster) over P(track)
    TH1F* fCemTrkPt_b[2];                  // Pt of leading track pointing to ele cluster
    TH1F* fCemTrkPtBc_b[2];                // beam-constrained Pt of leading track pointing to ele cluster
    TH1F* fCemTrkP_b[2];                   // P of leading track pointing to ele cluster
    TH1F* fCemTrkPBc_b[2];                 // beam-constrained P of leading track pointing to ele cluster
    TH1F* fCemTrkPhi_b[2];                 // Phi of leading track pointing to ele cluster
    TH1F* fCemTrkPhiBc_b[2];               // beam-constrained Phi of leading track pointing to ele cluster
    TH1F* fCemTrkTheta_b[2];               // Theta of leading track pointing to ele cluster
    TH1F* fCemTrkThetaBc_b[2];             // beam-constrained Theta of leading track pointing to ele cluster
    TH1F* fCemTrkEta_b[2];                 // Eta of leading track pointing to ele cluster
    TH1F* fCemTrkEtaBc_b[2];               // beam-constrained Eta of leading track pointing to ele cluster
    TH1F* fCemTrkDeltaX_b[2];              // dX=X(cluster)-X(extrapolated track)
    TH1F* fCemTrkDeltaXBc_b[2];            // dX=X(cluster)-X(extrapolated track), beam-constrained track
    TH1F* fCemTrkQDeltaX_b[2];              // dX*Q(track charge)
    TH1F* fCemTrkQDeltaXBc_b[2];            // dX*Q(track charge), beam-constrained track
    TH1F* fCemTrkDeltaZ_b[2];              // dZ=Z(cluster)-Z(extrapolated track)
    TH1F* fCemTrkDeltaZBc_b[2];            // dZ=Z(cluster)-Z(extrapolated track), beam-constrained track
    TH1F* fCemTrkZ0_b[2];                  // Z0(track)
    TH1F* fCemTrkZ0Bc_b[2];                // Z0(track), beam-constrained track
    TH1F* fCemTrkD0_b[2];                  // D0(track)
    TH1F* fCemTrkD0Bc_b[2];                // D0(track), beam-constrained track
    TH1F* fCemTrkCharge_b[2];              // Charge of track
    TH1F* fCemTrkRconv_b[2];               // Rconv of track (as calculated by MyTrackFilterModule)
    TH1F* fCemTrkAlgoID_b[2];              // Trk Algorithm ID
    TH1F* fCemTrkCotAx_b[2];               // number of COT axial superlayers
    TH1F* fCemTrkCotSt_b[2];               // number of COT stereo superlayers
    TH1F* fCemTrkCotAxHits_b[2];           // number of COT axial hits 
    TH1F* fCemTrkCotStHits_b[2];           // number of COT stereo hits 

    TH1F* fCemTrkCorrD0_b[2];              // D0 corrected for beam off-set
    TH1F* fCemTrkCorrDelPhi_b[2];          // effect of beam off-set correction: dPhi=CorrPhi-Phi


    //_______________________after cuts

    TH1F* fCemNcand_a;                  
    TH1F* fCemNtower_a[2];              
    TH1F* fCemEt_raw_a[2];              
    TH1F* fCemEt_corr_a[2];             
    TH1F* fCemEt_a[2];   
    TH1F* fCemE_em_a[2]; 
    TH1F* fCemE_total_a[2];             
    TH1F* fCemEta_det_a[2];             
    TH1F* fCemEta_a[2];                 
    TH1F* fCemPhi_a[2];                 
    TH1F* fCemTheta_a[2];               
    TH1F* fCemHadEm_a[2];               
    TH1F* fCemLshr_a[2];                
    TH1F* fCemCalIso4_raw_a[2];         
    TH1F* fCemCalIso4_corr_a[2];        
    TH1F* fCemCalIso4Fr_raw_a[2];       
    TH1F* fCemCalIso4Fr_corr_a[2];      
    TH1F* fCemChi2strip_a[2];           
    TH1F* fCemChi2_a[2];                
    TH1F* fCemFid_a[2];
    TH1F* fCemFidCes_a[2];             
    TH1F* fCemFidTrk_a[2];             
    TH1F* fCemEleXces_a[2];      
    TH1F* fCemTrkXces_a[2];      
    TH1F* fCemConvStat_a[2];                         
    TH1F* fCemEP_a[2];                  
    TH1F* fCemTrkPt_a[2];               
    TH1F* fCemTrkPtBc_a[2];             
    TH1F* fCemTrkP_a[2];                
    TH1F* fCemTrkPBc_a[2];              
    TH1F* fCemTrkPhi_a[2];              
    TH1F* fCemTrkPhiBc_a[2];            
    TH1F* fCemTrkTheta_a[2];            
    TH1F* fCemTrkThetaBc_a[2];          
    TH1F* fCemTrkEta_a[2];              
    TH1F* fCemTrkEtaBc_a[2];            
    TH1F* fCemTrkDeltaX_a[2];           
    TH1F* fCemTrkDeltaXBc_a[2];         
    TH1F* fCemTrkQDeltaX_a[2];          
    TH1F* fCemTrkQDeltaXBc_a[2];        
    TH1F* fCemTrkDeltaZ_a[2];           
    TH1F* fCemTrkDeltaZBc_a[2];         
    TH1F* fCemTrkZ0_a[2];               
    TH1F* fCemTrkZ0Bc_a[2];             
    TH1F* fCemTrkD0_a[2];               
    TH1F* fCemTrkD0Bc_a[2];             
    TH1F* fCemTrkCharge_a[2];           
    TH1F* fCemTrkRconv_a[2];            
    TH1F* fCemTrkAlgoID_a[2];           
    TH1F* fCemTrkCotAx_a[2];            
    TH1F* fCemTrkCotSt_a[2];            
    TH1F* fCemTrkCotAxHits_a[2];        
    TH1F* fCemTrkCotStHits_a[2];        

    TH1F* fCemTrkCorrD0_a[2];              
    TH1F* fCemTrkCorrDelPhi_a[2];          

  };

  //----------- histos for two leading plug electrons
  struct PEMElectron_t {
 
   // For plug electrons, electron energy is E = E(em cal) + E(ppr)*fUsePprEnergy 
   // unless otherwise noted 

    //_______________________before cuts

    TH1F* fPemNcand_b;                  // number of PEM candidates
    TH1F* fPemNtower_b[2];                 // number of towers in the cluster
    TH1F* fPemEt_raw_b[2];                 // raw Et(EM+HAD) with respect to vertex and PES
    TH1F* fPemEt_corr_b[2];                // Et correction (for tower response)
    TH1F* fPemEt_b[2];                     // corrected Et(EM) with respect to vertex and CES
    TH1F* fPemE_em_b[2];                   // corrected EM Energy
    TH1F* fPemE_ppr_b[2];                  // PPR energy 
    TH1F* fPemE_total_b[2];                // total (EM+HAD) corrected Energy
    TH1F* fPemEta_det_b[2];                // detector Eta
    TH1F* fPemEta_b[2];                    // event Eta 
    TH1F* fPemPhi_b[2];                    // Phi 
    TH1F* fPemTheta_b[2];                  // Theta 
    TH1F* fPemEta_pes3x3_b[2];             // eta of the pem 3x3chi2 fit
    TH1F* fPemPhi_pes3x3_b[2];             // phi of the pem 3x3chi2 fit
    TH1F* fPemHadEm_b[2];                  // Had/Em 
    TH1F* fPemCalIso4_raw_b[2];            // raw cal Iso in cone R=0.4 (sum Et in R=0.4)
    TH1F* fPemCalIso4_corr_b[2];           // cal Iso in cone R=0.4 corrected for leakage & multiple vert.
    TH1F* fPemCalIso4Fr_raw_b[2];          // raw CalIso4/EtEle
    TH1F* fPemCalIso4Fr_corr_b[2];         // corr. CalIso4/EtEle
    TH1F* fPem3x3FitTwr_b[2];              // .ne.0 if 3x3 fit was done 
    TH1F* fPemChi2Pes_b[2];                // PES 3x3 Chi2
    TH1F* fPemPes5x9U_b[2];                // 5x9 shower profile ratios (U)
    TH1F* fPemPes5x9V_b[2];                // 5x9 shower profile ratios (V)
    TH1F* fPemConvStat_b[2];             // conversion status (using regular tracks)
    TH1F* fPemTrkPt_b[2];                  // Pt of leading track pointing to ele cluster
    TH1F* fPemTrkPnxPt_b[2];               // Pt of leading Phoenix track pointing to ele cluster
    TH1F* fPemTrkPtBc_b[2];                // beam-constrained Pt of leading track pointing to ele cluster
    TH1F* fPemTrkP_b[2];                   // P of leading track pointing to ele cluster
    TH1F* fPemTrkPBc_b[2];                 // beam-constrained P of leading track pointing to ele cluster
    TH1F* fPemTrkPhi_b[2];                 // Phi of leading track pointing to ele cluster
    TH1F* fPemTrkPhiBc_b[2];               // beam-constrained Phi of leading track pointing to ele cluster
    TH1F* fPemTrkTheta_b[2];               // Theta of leading track pointing to ele cluster
    TH1F* fPemTrkThetaBc_b[2];             // beam-constrained Theta of leading track pointing to ele cluster
    TH1F* fPemTrkEta_b[2];                 // Eta of leading track pointing to ele cluster
    TH1F* fPemTrkEtaBc_b[2];               // beam-constrained Eta of leading track pointing to ele cluster
    TH1F* fPemTrkZ0_b[2];                  // Z0(track)
    TH1F* fPemTrkZ0Bc_b[2];                // Z0(track), beam-constrained track
    TH1F* fPemTrkD0_b[2];                  // D0(track)
    TH1F* fPemTrkD0Bc_b[2];                // D0(track), beam-constrained track
    TH1F* fPemTrkCharge_b[2];              // Charge of track
    TH1F* fPemTrkRconv_b[2];               // Rconv of track (as calculated by MyTrackFilterModule)
    TH1F* fPemTrkAlgoID_b[2];              // Trk Algorithm ID
    TH1F* fPemTrkPesDeltaR_b[2];           // PES-Trk delta R 
    TH1F* fPemPesDeltaR_b[2];              // PES-PEM delta R 
    TH1F* fPemTrkSiHits_b[2];              // number of Silicon hits 


    //_______________________after cuts
    TH1F* fPemNcand_a;                  
    TH1F* fPemNtower_a[2];              
    TH1F* fPemEt_raw_a[2];              
    TH1F* fPemEt_corr_a[2];             
    TH1F* fPemEt_a[2];                     
    TH1F* fPemE_em_a[2];                   
    TH1F* fPemE_ppr_a[2];                  
    TH1F* fPemE_total_a[2];             
    TH1F* fPemEta_det_a[2];             
    TH1F* fPemEta_a[2];                 
    TH1F* fPemPhi_a[2];                 
    TH1F* fPemTheta_a[2];               
    TH1F* fPemEta_pes3x3_a[2];          
    TH1F* fPemPhi_pes3x3_a[2];          
    TH1F* fPemHadEm_a[2];               
    TH1F* fPemCalIso4_raw_a[2];         
    TH1F* fPemCalIso4_corr_a[2];        
    TH1F* fPemCalIso4Fr_raw_a[2];       
    TH1F* fPemCalIso4Fr_corr_a[2];      
    TH1F* fPem3x3FitTwr_a[2];           
    TH1F* fPemChi2Pes_a[2];             
    TH1F* fPemPes5x9U_a[2];             
    TH1F* fPemPes5x9V_a[2];  
    TH1F* fPemConvStat_a[2];     
    TH1F* fPemTrkPt_a[2];              
    TH1F* fPemTrkPnxPt_a[2];        
    TH1F* fPemTrkPtBc_a[2];             
    TH1F* fPemTrkP_a[2];                
    TH1F* fPemTrkPBc_a[2];              
    TH1F* fPemTrkPhi_a[2];              
    TH1F* fPemTrkPhiBc_a[2];            
    TH1F* fPemTrkTheta_a[2];            
    TH1F* fPemTrkThetaBc_a[2];          
    TH1F* fPemTrkEta_a[2];                 
    TH1F* fPemTrkEtaBc_a[2];                
    TH1F* fPemTrkZ0_a[2];                  
    TH1F* fPemTrkZ0Bc_a[2];                
    TH1F* fPemTrkD0_a[2];                  
    TH1F* fPemTrkD0Bc_a[2];                
    TH1F* fPemTrkCharge_a[2];              
    TH1F* fPemTrkRconv_a[2];               
    TH1F* fPemTrkAlgoID_a[2];              
    TH1F* fPemTrkPesDeltaR_a[2];           
    TH1F* fPemPesDeltaR_a[2];           
    TH1F* fPemTrkSiHits_a[2];               

  };

  struct ZeeGeneral_t { //------------ General histograms for Z-->e+e-
                        // these histograms are filled if there is at least two ele
                        // in event
    //_______________________before cuts

    TH1F* fZeeNCEMtight_b; // number of tight CEM electrons 
    TH1F* fZeeNCEMloose_b; // number of loose CEM electrons 
    TH1F* fZeeNPEMtight_b; // number of tight PEM electrons 
    TH1F* fZeeNPEMloose_b; // number of loose PEM electrons 
    TH1F* fZeeNele_b;      // number of electrons which pass the cuts
    TH1F* fZeeNeleE_b;     // number of Extra electrons which pass the cuts but fail Pt cut

    TH1F* fZeeM_b;        // invariant mass of the Z candidate (using Et_ele & E_ele)
    TH1F* fZeeMtrk_b;     // invariant mass of two tracks from the Z candidate 
    TH1F* fZeeE_b;        // energy of the Z candidate 
    TH1F* fZeeEt_b;       // Pt of the Z candidate
    TH1F* fZeePhi_b;      // Phi of the Z candidate
    TH1F* fZeeEta_b;      // event Eta of the Z candidate
    TH1F* fZeeTheta_b;    // event Theta of the Z candidate
    TH1F* fZeeDeltaPhi_b; // delta Phi of two electrons 
    TH1F* fZeeDeltaEta_b; // delta Eta of two electrons 
    TH1F* fZeeDeltaR_b;   // delta R(eta,phi) of two electrons 

    TH1F* fZeeEleFid_b[2];   // fiduciality of the Z candidate electrons (0:CEM, 1:PEM)
    TH1F* fZeeEleStat_b[2];  // status of the Z candidate electrons (0:tight, 10:loose)
    TH1F* fZeeFid_b;         // fiduciality of the Z candidate: FidEle1+FidEle2 
                             //                 (0:CEM+CEM, 1:PEM+CEM, 2:PEM+PEM)
    TH1F* fZeeStat_b;        // status of the Z candidate: StatEle1+StatEle2 
                             // electron status (tCEM=0, lCEM=7, tPEM=9, lPEM=11)
    TH1F* fZeeQual_b;        // quality of the Z candidate: StatZ+FidZ
    TH1F* fZeeCharge_b;      // "charge" of the Z candidate: ChargeEle1+ChargeEle2
                             // ---> it has to be ZERO for true CEM+CEM Z's  

    TH1F* fZeeNele3_M_b;  // M_ee for events with Nele=3 (all 3 combinations)
    TH1F* fZeeNele4_M_b;  // M_ee for events with Nele=4 (all 6 combinations)
    TH2F* fZeeNele4_M1_vs_M2_b;  // Mee(pair1).vs.Mee(pair2) for events with Nele=4 (all 6 combinations)
    TH1F* fZeeNele3_M_Zlike_b;  // M_ee of a pair with min(|Mee-91.2|) for events with Nele=>3

    //_______________________after cuts
    TH1F* fZeeNCEMtight_a; 
    TH1F* fZeeNCEMloose_a; 
    TH1F* fZeeNPEMtight_a; 
    TH1F* fZeeNPEMloose_a; 
    TH1F* fZeeNele_a;      
    TH1F* fZeeNeleE_a;     

    TH1F* fZeeM_a;
    TH1F* fZeeMtrk_a;
    TH1F* fZeeE_a;        
    TH1F* fZeeEt_a;       
    TH1F* fZeePhi_a;      
    TH1F* fZeeEta_a;      
    TH1F* fZeeTheta_a;
    TH1F* fZeeDeltaPhi_a;
    TH1F* fZeeDeltaEta_a;
    TH1F* fZeeDeltaR_a;  
    TH1F* fZeeEleFid_a[2];   
    TH1F* fZeeEleStat_a[2];  
    TH1F* fZeeFid_a;                                      
    TH1F* fZeeStat_a;                                     
    TH1F* fZeeQual_a;        
    TH1F* fZeeCharge_a;      

    TH1F* fZeeNele3_M_a;
    TH1F* fZeeNele4_M_a;
    TH2F* fZeeNele4_M1_vs_M2_a;
    TH1F* fZeeNele3_M_Zlike_a;
  };

  struct ConversionHisto_t {
    //_______________________________________ 
    //***** histos for conversion photons
    TH1F* fConvNcand[2]; // number of conversion candidates for each electron
    TH1F* fConvR[2];     // radius of conversion for best conversion candidate
    TH1F* fConvXYsep[2]; // XY separation for best conversion candidate
    TH1F* fConvDelLam[2];// deltaCot(theta) for best conversion candidate 
    TH1F* fConvChi2[2];  // my convChi^2 for best conversion candidate
    TH1F* fConvDeltaR[2];   // dR=myRconv1-myRconv2 for best conversion candidate  
    TH1F* fConvTrkAlgoID[2];// algorithm ID for best conversion candidate
    TH1F* fConvMyRconv[2];// my conversion Radius for best conversion candidate
    TH1F* fConvD0[2];// impact parameter of conversion photon
    TH1F* fConvEleInd[2]; // ind+1 of matching electron for best conversion candidate
    TH1F* fConvMyRconv1R;   // myRconv("parent ele")-R(pair)
    TH1F* fConvMyRconv2R;   // myRconv("daughter ele")-R(pair)
    TH1F* fConvMyRconvEle;  // My Rconv for original electron
    TH1F* fConvDeltaPhi[2]; // dPhi of conversion photon and an electron

    TH1F* fConvTriD0; // impact parameter of conversion photon for 2nd e+e- pair from tridents
    TH1F* fConvTriR;  // conversion Radius for 2nd e+e- pair from tridents
    TH1F* fConvTriMyRconv; // my conversion Radius for 3rd electron from from tridents
    TH1F* fConvTriDeltaR; // dR=myRconv2-myRconv3 for 2nd e+e- pair from tridents
  };


//----------------------------------------------------------------------------------------
//  *****  Declaring output parameters/results of the ZeeFilter Module for ONE event ***** 
//________________________________________________________________________________________
  struct ZeeOutput {    
    int outNCEMtight; // number of CEM ele passing tight cuts 
    int outNCEMloose; // number of CEM ele passing loose cuts (excluding tight ele) 
    int outNPEMtight; // number of PEM ele passing tight cuts
    int outNPEMloose; // number of PEM ele passing loose cuts (excluding tight ele)
    int outNele; // total number of electrons passing loose & tight cuts     
    int outNeleL; // total number of electrons passing ID cuts and Pt-cut 
    int outNeleE; // total number of extra electrons passing cuts (except Pt-cut)
    TLorentzVector outMyZee;      // my Zee using ele
    TLorentzVector outMyZeeTrk;   // my Zee using trk
    std::vector<TLorentzVector> outMyEle;      // my corrected electrons (Em(CEM) and Em+Ppr(PEM) energy only)
    std::vector<TLorentzVector> outMyEleRaw;   // my uncorrected electrons (corresponding matched raw photon) !!!
    std::vector<TLorentzVector> outMyEleTrk;   // my Zee electron tracks
    std::vector<int> outTrkInd;  // track number in TrackBlock and PhoenixTrackBlock 
    std::vector<int> outEleInd;  // electron index in TStnElectron block 
    std::vector<int> outEleFid;  // electron fiduciality (CEM=0, PEM=1)
    std::vector<int> outEleStat;  // electron status (tCEM=0, lCEM=7, tPEM=9, lPEM=11)
    std::vector<int> outEleConvStat;  // electron conversion status (prompt=0, conversion=1, "trident"=-1)
    std::vector<int> outEleCharge;   // electron charge (it can be 0 for plug electrons)
    std::vector<double> outEleDetEta; // electron detector eta
    std::vector<double> outEleClusPhi; // electron detector phi (phi of EM cluster)
    std::vector<double> outEleIso; // corrected electron ISO
    std::vector<double> outEleIsoRaw; // raw electron ISO
    std::vector<double> outEleChi2; // electron Chi2
    std::vector<double> outEleHadEm; // electron Had/Em
    std::vector<double> outEleCprPpr; // electron CPR/PPR energy 
    std::vector<double> outEleXces; // electron X_ces position
    std::vector<double> outEleZces; // electron Z_ces position

    int outZeeFid; // FidZee=FidEle1+FidEle2      
    int outZeeStat;  // StatZee=StatEle1+StatEle2 
    int outZeeQual;  // QualZee=FidZee+StatZee 
    int outZeeCharge; // Q(ele1)+Q(ele2)
  };

//----------------------------------------------------------------------------------------
//  *****  Declaring output parameters/results of the Conversion finder ***** 
//________________________________________________________________________________________
  struct ConversionOutput {
    int conv_Ncand[2]; // number of conversion candidates for ele1,2
    //    int conv_Ntrident[2]; // number of trident candidates for a pair track of ele1,2
    double conv_Chi2[2]; // my Chi2 of best conversion candidates for ele1,2 
    double conv_dXY[2]; // XY sep of best conversion candidates for ele1,2
    double conv_dCot[2]; // delta Cot(theta) of best conversion candidates for ele1,2
    double conv_R[2]; // pair conversion radius of best conversion candidates for ele1,2
    double conv_myRconv[2]; // my conversion radius of best conversion candidates for ele1,2
    double conv_dmyRconv[2]; // myRconv1-myRconv2 of best conversion candidates for ele1,2
    int conv_ConvInd[2]; // Trk index in StnTrackBlock of best conversion candidates for ele1,2
    int conv_TrkAlgo[2]; // Track algorithm of best conversion candidates for ele1,2
    int conv_EleInd[2]; // ind+1 of matching ele for best conversion candidates for ele1,2
    double conv_D0[2]; // impact parameter of conversion photon
    double conv_dPhi[2]; // dPhi of conversion photon and an electron track of interest
    double tri_D0[2]; // impact parameter of a soft photon from trident events
    double tri_R[2];  // conversion radius of e+e- pair from trident events
    double tri_myRconv[2]; // my conversion radius for 3rd electron from trident events
    
    TLorentzVector conv_pho1; // 4-vec of photon corresponding to ele1
    TLorentzVector conv_pho2; // 4-vec of photon corresponding to ele2   
  };


  enum { myFirstRun = 138809 };  // need this to get corrections for beam off-set
  enum { myLastRun = 264071 };
  enum { myNruns = myLastRun - myFirstRun + 1};

protected:

  char histo_outputstring[200];

  //------------------------------------- need this for beam off-set correction
  struct MyBeamLine {
    double bm_x[myNruns];
    double bm_y[myNruns];
    double bmsl_x[myNruns];
    double bmsl_y[myNruns];    
  } fBeam;

  TVector3 myBeam;
  TVector3 myBeamSlope;
  char* fMyBeamLineFile;
  int fUseBeamLineDB;  // switch to use beamline DB or a beam line file

  int EventCount_b; // global event count before cuts
  int EventCount_a; // global event count after cuts
  //____________________________________________________________________________


                           // pointers to the data blocks used
  TStnElectronBlock* fElectronBlock;
  TStnPhotonBlock* fPhotonBlock;
  TStnTrackBlock*  fTrackBlock;
  TPhoenixElectronBlock* fPhxEleBlock; // phoenix stuff
  TStnTrackBlock* fPhxSiTrackBlock; // phoenix stuff
    
  TStnBeamPosBlock* fCotBeamPosBlock; // COT beam position
  TStnBeamPosBlock* fSvxBeamPosBlock; // Silicon beam position

                           // histograms 
  ElectronGeneral_t fEleHist;
  CEMElectron_t fCEMHist;
  PEMElectron_t fPEMHist;
  ZeeGeneral_t fZeeHist;
  ConversionHisto_t fConvHist; // all conversions

  ZeeOutput myZeeStuff; // default output params
  ConversionOutput myConvStuff; // conversion finder output params
  
  //___________________________________________ cuts & service params
  int fUseConvCut; // turn On/Off conversion cut in eleID
  int fUseZeeCut; // turn Z->ee (=1) or InclusiveEle (=0) modes
  int fUseMyExtraCemCut; // switch to remove mis-measured CEM electrons

  //***** cuts for CEM electrons

  double fCemMinEt; // min Et of CEM electrons
  double fCemMaxEt; // max Et of CEM electrons
  double fTrkMinPt; // min Pt of Tracks corresponding to CEM electrons
  double fTrkMaxPt; // max Pt of Tracks corresponding to CEM electrons

  double fCemMinHADEM[2]; // HADEM cut using 3towers : [0]-const term in sliding cut
  double fCemMaxHADEM[2];                      // **** [1]-linear term in sliding cut
  double fCemMinIsoFr;  // Iso fraction: CorrIso(R=0.4)/Et
  double fCemMaxIsoFr;  
  double fCemMinLshr;   // cut on Lshr 
  double fCemMaxLshr;
  double fCemMinEP;   // E over P cut 
  double fCemMaxEP;
  double fCemMinChi2strip; // CES strip Chi2 cut
  double fCemMaxChi2strip;
  double fCemMindXQ; // signed X(trk)-X(ces) cut: Q*(X_trk-X_ces)
  double fCemMaxdXQ; // using beam-constrained track
  double fCemMinCesdZ; // cut on dZ=Z(cluster)-Z(extrapolated track), beam-constrained track
  double fCemMaxCesdZ;
  double fCemMinTrkZ0; // cut on Trk Z0, beam-constrained track
  double fCemMaxTrkZ0;
  int fCemFidCut; // cut on CEM fiduciality
  int fCemConvCut; // cut on Conversion status code
  int fCemTrkMatch; // to be used to filter out events with TrkInd=-1
  int fCemMinCotAxSl; // cut on number of COT Axial SL with 5 or more hits 
  int fCemMaxCotAxSl;
  int fCemMinCotStSl; // cut on number of COT Stereo SL with 5 or more hits 
  int fCemMaxCotStSl;
  double fCemMinRconv; // cut on Rconv as defined in TMyTrackFilterModule
  double fCemMaxRconv;

  //_______________________________________________________________________
  //***** conversion cuts
  double fConv_dXYmax; // cut on max 2D separation of two conversion candidates
  double fConv_dCotMax; // cut on cot(theta1)-cot(theta2) of two conversion candidates
  double fConv_Rmax; // cut on max "mutual" Rconversion of two conversion candidates   
  int fCovCotAxSegMin; // cut on the min number of CotAxSeg for conversion candidate 
  int fCovCotStSegMin; // cut on the min number of CotStSeg for conversion candidate 

  //_______________________________________________________________________
  //***** cuts for PEM electrons
  double fPemMinEt; // min Et of PEM electrons
  double fPemMaxEt; // max Et of PEM electrons
  double fPemMinEtaPes; // min PES Eta of PEM electrons
  double fPemMaxEtaPes; 
  double fPemMinHADEM; // HADEM cut: const term only for PEM
  double fPemMaxHADEM;      
  double fPemMin3x3Chi2;  // cut on PES 3x3 fit Chi2
  double fPemMax3x3Chi2;
  double fPemMinPes5x9U; // cut on 5x9 shower profile ratios (U)
  double fPemMaxPes5x9U; 
  double fPemMinPes5x9V; // cut on 5x9 shower profile ratios (V)
  double fPemMaxPes5x9V; 
  double fPemMinIsoFr;  // cut on Iso fraction: CorrIso(R=0.4)/Et
  double fPemMaxIsoFr;  
  int fPem3x3FitCut; // cut on PEM 3x3 tower fit status code
  int fPemTrkCut;    // cut on Trk status code
  int fPemPhnxMatchCut; // cut on Phoenix track match
  double fPemMinPesdR; // cut on dR=R(PES)-R(extrapolated track)
  double fPemMaxPesdR;
  double fPemMinTrkZ0; // cut on Trk Z0
  double fPemMaxTrkZ0;
  double fPemMinPhnxZ0; // cut on Phoenix Z0
  double fPemMaxPhnxZ0;
  int fPemMinSiHit;  // cut on number of Silicon hits 
  int fPemMaxSiHit;

  //_______________________________________________________________________
  //***** cuts for Z candidates
  int fMinNele; // cut on number of leading+extra electrons which pass ID cuts 
  int fMaxNele; // both CEM and PEM, tight and loose  
  int fMinNeleL; // cut on number of leading electrons which pass ID cuts 
  int fMaxNeleL; // both CEM and PEM, tight and loose
  int fMinNeleE; // cut on number of extra electron candidates which pass ID cuts 
  int fMaxNeleE; // and have energy in the range Et_extra < Et < MinEt_leading 
  double fEleEt_leading; // cut on Et of leading electrons

  double fZeeMinM; // cut on invariant mass of the Z candidate
  double fZeeMaxM;
  double fZeeMinEt; // cut on Et of the Z candidate
  double fZeeMaxEt;
  double fZeeMinEta; // cut on event Eta of the Z candidate
  double fZeeMaxEta;
  double fZeeMinDeltaPhi; // cut on dPhi of two electrons from the Z candidate
  double fZeeMaxDeltaPhi;
  double fZeeMinDeltaEta; // cut on dEta of two electrons from the Z candidate
  double fZeeMaxDeltaEta;
  double fZeeMinDeltaR; // cut on dR of two electrons from the Z candidate
  double fZeeMaxDeltaR;
  int fZeeMinFid; // cut on fiduciality of the Z candidate: FidEle1+FidEle2 
  int fZeeMaxFid; //                 (0:CEM+CEM, 1:PEM+CEM, 2:PEM+PEM)
  int fZeeMinStat; // cut on status of the Z candidate: StatEle1+StatEle2 
  int fZeeMaxStat; // electron status (tCEM=0, lCEM=7, tPEM=9, lPEM=11)
  int fZeeMinQual; // cut on quality of the Z candidate: StatZ+FidZ
  int fZeeMaxQual;
  int fZeeMinCharge; // cut on "charge" of the Z candidate: ChargeEle1+ChargeEle2
  int fZeeMaxCharge;                   

  int fUseVerbose; // service param

  char* fDatFileName;

public:

  TMyZeeFilterModule(const char* name="MyZee", 
		     const char* title="MyZee");
  ~TMyZeeFilterModule();
                                   // ***** accessors
  
  ElectronGeneral_t*  GetEleHist() { return &fEleHist; }
  CEMElectron_t*      GetCEMHist() { return &fCEMHist; }
  PEMElectron_t*      GetPEMHist() { return &fPEMHist; }
  ZeeGeneral_t*       GetZeeHist() { return &fZeeHist; }
  ConversionHisto_t*  GetConvHist() { return &fConvHist; }

  ZeeOutput*          GetZeeOutput() { return &myZeeStuff; }
  ConversionOutput*   GetConvOutput() { return &myConvStuff; } 


  TStnElectronBlock*  GetElectronBlock() { return fElectronBlock; }
  TStnPhotonBlock*    GetPhotonBlock() { return fPhotonBlock; }
  TStnTrackBlock*     GetTrackBlock()    { return fTrackBlock; }
  TPhoenixElectronBlock* GetPhxEleBlock() { return fPhxEleBlock; } // phoenix stuff
  TStnTrackBlock* GetPhxSiTrackBlock() { return fPhxSiTrackBlock; } // phoenix stuff

  TStnBeamPosBlock* GetCotBeamPosBlock() { return fCotBeamPosBlock; }
  TStnBeamPosBlock* GetSvxBeamPosBlock() { return fSvxBeamPosBlock; }


  //-------- part related to beam off-set correction --------------
  MyBeamLine* GetMyBeam()         { return &fBeam; } 
  char*  GetMyBeamLineFileName() { return fMyBeamLineFile; }
  void SetMyBeamLineFileName(char* fname) { fMyBeamLineFile = fname; }    
  double myPhi0_corr(TVector3 beam, TVector3 slope, TStnTrack* Trk);
  double myD0_corr(TVector3 beam, TVector3 slope, TStnTrack* Trk);
  void InitMyBeamLineOld();
  void InitMyBeamLineNew(int RunNumber);
  void GetMyBeamLine(TVector3 &beam, TVector3 &slope, int myRun);
  int GetUseBeamLineDB() { return fUseBeamLineDB; }
  void SetUseBeamLineDB(int cut) { fUseBeamLineDB = cut; }
  //_______________________________________________________________

  int GetUseConvCut() { return fUseConvCut; }
  int GetUseZeeCut() { return fUseZeeCut; }
                               //***** accessors for CEM electron cuts
  double GetCemMinEt() { return fCemMinEt; }
  double GetCemMaxEt() { return fCemMaxEt; }
  double GetTrkMinPt() { return fTrkMinPt; }
  double GetTrkMaxPt() { return fTrkMaxPt; }
  double GetCemMinHADEM(int ind) { return fCemMinHADEM[ind]; }
  double GetCemMaxHADEM(int ind) { return fCemMaxHADEM[ind]; }
  double GetCemMinIsoFr() { return fCemMinIsoFr; } 
  double GetCemMaxIsoFr() { return fCemMaxIsoFr; } 
  double GetCemMinLshr() { return fCemMinLshr; }  
  double GetCemMaxLshr() { return fCemMaxLshr; }
  double GetCemMinEP() { return fCemMinEP; } 
  double GetCemMaxEP() { return fCemMaxEP; }
  double GetCemMinChi2strip() { return fCemMinChi2strip; }
  double GetCemMaxChi2strip() { return fCemMaxChi2strip; }
  double GetCemMindXQ() { return fCemMindXQ; }
  double GetCemMaxdXQ() { return fCemMaxdXQ; }
  double GetCemMinCesdZ() { return fCemMinCesdZ; } 
  double GetCemMaxCesdZ() { return fCemMaxCesdZ; }
  double GetCemMinTrkZ0() { return fCemMinTrkZ0; }
  double GetCemMaxTrkZ0() { return fCemMaxTrkZ0; }
  int GetCemFidCut() { return fCemFidCut; } 
  int GetCemTrkMatch() { return fCemTrkMatch; }
  int GetCemConvCut() { return fCemConvCut; } 
  int GetCemMinCotAxSl() { return fCemMinCotAxSl; } 
  int GetCemMaxCotAxSl() { return fCemMaxCotAxSl; }
  int GetCemMinCotStSl() { return fCemMinCotStSl; }
  int GetCemMaxCotStSl() { return fCemMaxCotStSl; }
  double GetCemMinRconv() { return fCemMinRconv; }
  double GetCemMaxRconv() { return fCemMaxRconv; }


  //_______________________________________________________________________
  //***** accessors to conversion cuts
  double GetConv_dXYmax() { return fConv_dXYmax; } 
  double GetConv_dCotMax() { return fConv_dCotMax; } 
  double GetConv_Rmax() { return fConv_Rmax; }
  int GetCovCotAxSegMin() { return fCovCotAxSegMin; }  
  int GetCovCotStSegMin() { return fCovCotStSegMin; } 


  //_______________________________________________________________________
                               //***** accessors for PEM electron cuts
  double GetPemMinEt() { return fPemMinEt; }
  double GetPemMaxEt() { return fPemMaxEt; }
  double GetPemMinEtaPes() { return fPemMinEtaPes; }
  double GetPemMaxEtaPes() { return fPemMaxEtaPes; }
  double GetPemMinHADEM() { return fPemMinHADEM; }
  double GetPemMaxHADEM() { return fPemMaxHADEM; }
  double GetPemMin3x3Chi2() { return fPemMin3x3Chi2; } 
  double GetPemMax3x3Chi2() { return fPemMax3x3Chi2; }
  double GetPemMinPes5x9U() { return fPemMinPes5x9U; }
  double GetPemMaxPes5x9U() { return fPemMaxPes5x9U; }
  double GetPemMinPes5x9V() { return fPemMinPes5x9V; }
  double GetPemMaxPes5x9V() { return fPemMaxPes5x9V; }
  double GetPemMinIsoFr() { return fPemMinIsoFr; }
  double GetPemMaxIsoFr() { return fPemMaxIsoFr; } 
  int GetPem3x3FitCut() { return fPem3x3FitCut; }
  int GetPemTrkCut() { return fPemTrkCut; }  
  int GetPemPhnxMatchCut() { return fPemPhnxMatchCut; } 
  double GetPemMinPesdR() { return fPemMinPesdR; }
  double GetPemMaxPesdR() { return fPemMaxPesdR; }
  double GetPemMinTrkZ0() { return fPemMinTrkZ0; }
  double GetPemMaxTrkZ0() { return fPemMaxTrkZ0; }
  double GetPemMinPhnxZ0() { return fPemMinPhnxZ0; } 
  double GetPemMaxPhnxZ0() { return fPemMaxPhnxZ0; }
  int GetPemMinSiHit() { return fPemMinSiHit; }
  int GetPemMaxSiHit() { return fPemMaxSiHit; }

  //_______________________________________________________________________
  //***** accessors to cuts on Z candidates
  int GetMinNele() { return fMinNele; }
  int GetMaxNele() { return fMaxNele; }
  int GetMinNeleL() { return fMinNeleL; }
  int GetMaxNeleL() { return fMaxNeleL; }
  int GetMinNeleE() { return fMinNeleE; }
  int GetMaxNeleE() { return fMaxNeleE; }
  double GetEleEt_leading() { return fEleEt_leading; } 
  double GetZeeMinM() { return fZeeMinM; } 
  double GetZeeMaxM() { return fZeeMaxM; }
  double GetZeeMinEt() { return fZeeMinEt; } 
  double GetZeeMaxEt() { return fZeeMaxEt; }
  double GetZeeMinEta() { return fZeeMinEta; } 
  double GetZeeMaxEta() { return fZeeMaxEta; }
  double GetZeeMinDeltaPhi() { return fZeeMinDeltaPhi; } 
  double GetZeeMaxDeltaPhi() { return fZeeMaxDeltaPhi; } 
  double GetZeeMinDeltaEta() { return fZeeMinDeltaEta; } 
  double GetZeeMaxDeltaEta() { return fZeeMaxDeltaEta; } 
  double GetZeeMinDeltaR() { return fZeeMinDeltaR; } 
  double GetZeeMaxDeltaR() { return fZeeMaxDeltaR; } 
  int GetZeeMinFid() { return fZeeMinFid; }
  int GetZeeMaxFid() { return fZeeMaxFid; }
  int GetZeeMinStat() { return fZeeMinStat; } 
  int GetZeeMaxStat() { return fZeeMaxStat; }
  int GetZeeMinQual() { return fZeeMinQual; }
  int GetZeeMaxQual() { return fZeeMaxQual; }
  int GetZeeMinCharge() { return fZeeMinCharge; } 
  int GetZeeMaxCharge() { return fZeeMaxCharge; }                  

  //__________________________________________________________ accessors to the
  //__________________________________________________________ output params
  int GetmyNCEMtight() { return myZeeStuff.outNCEMtight; }
  int GetmyNCEMloose() { return myZeeStuff.outNCEMloose; }
  int GetmyNPEMtight() { return myZeeStuff.outNPEMtight; }
  int GetmyNPEMloose() { return myZeeStuff.outNPEMloose; }
  int GetmyNele() { return myZeeStuff.outNele; }     
  int GetmyNeleL() { return myZeeStuff.outNeleL; }     
  int GetmyNeleE() { return myZeeStuff.outNeleE; }     

  double GetmyZeeM() { return myZeeStuff.outMyZee.M(); }
  double GetmyZeeE() { return myZeeStuff.outMyZee.E(); } 
  double GetmyZeeEt() { return myZeeStuff.outMyZee.Pt(); }  
  double GetmyZeePhi() { return myZeeStuff.outMyZee.Phi(); } 
  double GetmyZeeTheta() { return myZeeStuff.outMyZee.Theta(); } 
  double GetmyZeeEta() { return myZeeStuff.outMyZee.Eta(); } 
  double GetmyZeeM_trk() { return myZeeStuff.outMyZeeTrk.M(); }
  double GetmyZeeE_trk() { return myZeeStuff.outMyZeeTrk.E(); } 
  double GetmyZeeEt_trk() { return myZeeStuff.outMyZeeTrk.Pt(); } 

  TLorentzVector* GetmyZee() { return &myZeeStuff.outMyZee; }
  TLorentzVector* GetmyZeeTrk() { return &myZeeStuff.outMyZeeTrk; }
  int GetmyZeeFid() { return myZeeStuff.outZeeFid; }                     
  int GetmyZeeStat() { return myZeeStuff.outZeeStat; }                       
  int GetmyZeeQual() { return myZeeStuff.outZeeQual; } 
  int GetmyZeeCharge() { return myZeeStuff.outZeeCharge; } 

  int GetmyZeeTrkInd(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outTrkInd.size()) return myZeeStuff.outTrkInd[i];
    else return -1;
  }
  int GetmyZeeEleInd(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleInd.size()) return myZeeStuff.outEleInd[i];
    else return -1;
  }
  int GetmyEleFid(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleFid.size()) return myZeeStuff.outEleFid[i];
    else return -1;
  }
  int GetmyEleStat(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleStat.size()) return myZeeStuff.outEleStat[i];
    else return -1;
  }
  int GetmyEleCharge(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleCharge.size()) return myZeeStuff.outEleCharge[i];
    else return -10;
  }
  int GetmyEleConvStat(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleConvStat.size()) return myZeeStuff.outEleConvStat[i];
    else return 0;
  }
  double GetmyEleDetEta(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleDetEta.size()) return myZeeStuff.outEleDetEta[i];
    else return -999.9;
  }
  double GetmyEleClusPhi(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleClusPhi.size()) return myZeeStuff.outEleClusPhi[i];
    else return -999.9;
  }
  double GetmyEleIso(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleIso.size()) return myZeeStuff.outEleIso[i];
    else return -999.9;
  }
  double GetmyEleIsoRaw(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleIsoRaw.size()) return myZeeStuff.outEleIsoRaw[i];
    else return -999.9;
  }
  double GetmyEleChi2(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleChi2.size()) return myZeeStuff.outEleChi2[i];
    else return -999.9;
  }
  double GetmyEleHadEm(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleHadEm.size()) return myZeeStuff.outEleHadEm[i];
    else return -999.9;
  }
  double GetmyEleCprPpr(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleCprPpr.size()) return myZeeStuff.outEleCprPpr[i];
    else return -999.9;
  }
  double GetmyEleXces(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleXces.size()) return myZeeStuff.outEleXces[i];
    else return -999.9;
  }
  double GetmyEleZces(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outEleZces.size()) return myZeeStuff.outEleZces[i];
    else return -999.9;
  }
  TLorentzVector* GetmyEle(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outMyEle.size()) return &myZeeStuff.outMyEle[i]; 
    else return NULL;
  }
  TLorentzVector* GetmyEleRaw(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outMyEleRaw.size()) return &myZeeStuff.outMyEleRaw[i]; 
    else return NULL;
  }
  TLorentzVector* GetmyEleTrk(int i) 
  { 
    if(i>=0 && i<myZeeStuff.outMyEleTrk.size()) return &myZeeStuff.outMyEleTrk[i]; 
    else return NULL;
  }


  int   GetUseVerbose()  { return fUseVerbose; }
  char* GetDatFileName()  { return fDatFileName; }


  //________________________ cut setters ____________________________________________

  void SetUseConvCut(int cut) { fUseConvCut = cut; }
  void SetUseZeeCut(int cut)  { fUseZeeCut = cut; }
  void SetUseMyExtraCemCut(int cut) { fUseMyExtraCemCut=cut; }

                               //***** setters for CEM electron cuts
  void SetCemMinEt(double cut) { fCemMinEt=cut; }
  void SetCemMaxEt(double cut) { fCemMaxEt=cut; }
  void SetTrkMinPt(double cut) { fTrkMinPt=cut; }
  void SetTrkMaxPt(double cut) { fTrkMaxPt=cut; }
  void SetCemMinHADEM(int ind, double cut) { fCemMinHADEM[ind]=cut; }
  void SetCemMaxHADEM(int ind, double cut) { fCemMaxHADEM[ind]=cut; }
  void SetCemMinIsoFr(double cut) { fCemMinIsoFr=cut; } 
  void SetCemMaxIsoFr(double cut) { fCemMaxIsoFr=cut; } 
  void SetCemMinLshr(double cut) { fCemMinLshr=cut; }  
  void SetCemMaxLshr(double cut) { fCemMaxLshr=cut; }
  void SetCemMinEP(double cut) { fCemMinEP=cut; } 
  void SetCemMaxEP(double cut) { fCemMaxEP=cut; }
  void SetCemMinChi2strip(double cut) { fCemMinChi2strip=cut; }
  void SetCemMaxChi2strip(double cut) { fCemMaxChi2strip=cut; }
  void SetCemMindXQ(double cut) { fCemMindXQ=cut; }
  void SetCemMaxdXQ(double cut) { fCemMaxdXQ=cut; }
  void SetCemMinCesdZ(double cut) { fCemMinCesdZ=cut; } 
  void SetCemMaxCesdZ(double cut) { fCemMaxCesdZ=cut; }
  void SetCemMinTrkZ0(double cut) { fCemMinTrkZ0=cut; }
  void SetCemMaxTrkZ0(double cut) { fCemMaxTrkZ0=cut; }
  void SetCemFidCut(int cut) { fCemFidCut=cut; } 
  void SetCemConvCut(int cut) { fCemConvCut=cut; } 
  void SetCemTrkMatch(int cut) { fCemTrkMatch=cut; }
  void SetCemMinCotAxSl(int cut) { fCemMinCotAxSl=cut; } 
  void SetCemMaxCotAxSl(int cut) { fCemMaxCotAxSl=cut; }
  void SetCemMinCotStSl(int cut) { fCemMinCotStSl=cut; }
  void SetCemMaxCotStSl(int cut) { fCemMaxCotStSl=cut; }
  void SetCemMinRconv(double cut) { fCemMinRconv=cut; }
  void SetCemMaxRconv(double cut) { fCemMaxRconv=cut; }

  //_______________________________________________________________________
                               //***** setters to conversion cuts
  void SetConv_dXYmax(double cut) { fConv_dXYmax=cut; } 
  void SetConv_dCotMax(double cut) { fConv_dCotMax=cut; } 
  void SetConv_Rmax(double cut) { fConv_Rmax=cut; }
  void SetCovCotAxSegMin(int cut) { fCovCotAxSegMin=cut; }  
  void SetCovCotStSegMin(int cut) { fCovCotStSegMin=cut; } 

  //_______________________________________________________________________
                               //***** setters for PEM electron cuts
  void SetPemMinEt(double cut) { fPemMinEt=cut; }
  void SetPemMaxEt(double cut) { fPemMaxEt=cut; }
  void SetPemMinEtaPes(double cut) { fPemMinEtaPes=cut; }
  void SetPemMaxEtaPes(double cut) { fPemMaxEtaPes=cut; }
  void SetPemMinHADEM(double cut) { fPemMinHADEM=cut; }
  void SetPemMaxHADEM(double cut) { fPemMaxHADEM=cut; }
  void SetPemMin3x3Chi2(double cut) { fPemMin3x3Chi2=cut; } 
  void SetPemMax3x3Chi2(double cut) { fPemMax3x3Chi2=cut; }
  void SetPemMinPes5x9U(double cut) { fPemMinPes5x9U=cut; }
  void SetPemMaxPes5x9U(double cut) { fPemMaxPes5x9U=cut; }
  void SetPemMinPes5x9V(double cut) { fPemMinPes5x9V=cut; }
  void SetPemMaxPes5x9V(double cut) { fPemMaxPes5x9V=cut; }
  void SetPemMinIsoFr(double cut) { fPemMinIsoFr=cut; }
  void SetPemMaxIsoFr(double cut) { fPemMaxIsoFr=cut; } 
  void SetPem3x3FitCut(int cut) { fPem3x3FitCut=cut; }
  void SetPemTrkCut(int cut) { fPemTrkCut=cut; }  
  void SetPemPhnxMatchCut(int cut) { fPemPhnxMatchCut=cut; } 
  void SetPemMinPesdR(double cut) { fPemMinPesdR=cut; }
  void SetPemMaxPesdR(double cut) { fPemMaxPesdR=cut; }
  void SetPemMinTrkZ0(double cut) { fPemMinTrkZ0=cut; }
  void SetPemMaxTrkZ0(double cut) { fPemMaxTrkZ0=cut; }
  void SetPemMinPhnxZ0(double cut) { fPemMinPhnxZ0=cut; } 
  void SetPemMaxPhnxZ0(double cut) { fPemMaxPhnxZ0=cut; }
  void SetPemMinSiHit(int cut) { fPemMinSiHit=cut; }
  void SetPemMaxSiHit(int cut) { fPemMaxSiHit=cut; }

  //_______________________________________________________________________
  //***** accessors to cuts on Z candidates
  void SetMinNele(int cut) { fMinNele=cut; }
  void SetMaxNele(int cut) { fMaxNele=cut; }
  void SetMinNeleL(int cut) { fMinNeleL=cut; }
  void SetMaxNeleL(int cut) { fMaxNeleL=cut; }
  void SetMinNeleE(int cut) { fMinNeleE=cut; }
  void SetMaxNeleE(int cut) { fMaxNeleE=cut; }
  void SetEleEt_leading(double cut) { fEleEt_leading=cut; } 
  void SetZeeMinM(double cut) { fZeeMinM=cut; } 
  void SetZeeMaxM(double cut) { fZeeMaxM=cut; }
  void SetZeeMinEt(double cut) { fZeeMinEt=cut; } 
  void SetZeeMaxEt(double cut) { fZeeMaxEt=cut; }
  void SetZeeMinEta(double cut) { fZeeMinEta=cut; } 
  void SetZeeMaxEta(double cut) { fZeeMaxEta=cut; }
  void SetZeeMinDeltaPhi(double cut) { fZeeMinDeltaPhi=cut; } 
  void SetZeeMaxDeltaPhi(double cut) { fZeeMaxDeltaPhi=cut; } 
  void SetZeeMinDeltaEta(double cut) { fZeeMinDeltaEta=cut; } 
  void SetZeeMaxDeltaEta(double cut) { fZeeMaxDeltaEta=cut; } 
  void SetZeeMinDeltaR(double cut) { fZeeMinDeltaR=cut; } 
  void SetZeeMaxDeltaR(double cut) { fZeeMaxDeltaR=cut; } 
  void SetZeeMinFid(int cut) { fZeeMinFid=cut; }
  void SetZeeMaxFid(int cut) { fZeeMaxFid=cut; }
  void SetZeeMinStat(int cut) { fZeeMinStat=cut; } 
  void SetZeeMaxStat(int cut) { fZeeMaxStat=cut; }
  void SetZeeMinQual(int cut) { fZeeMinQual=cut; }
  void SetZeeMaxQual(int cut) { fZeeMaxQual=cut; }
  void SetZeeMinCharge(int cut) { fZeeMinCharge=cut; } 
  void SetZeeMaxCharge(int cut) { fZeeMaxCharge=cut; }                  

  //__________________________________________________________ setters of the
  //__________________________________________________________ output params

  void SetmyNCEMtight(int param) { myZeeStuff.outNCEMtight=param; }
  void SetmyNCEMloose(int param) { myZeeStuff.outNCEMloose=param; }
  void SetmyNPEMtight(int param) { myZeeStuff.outNPEMtight=param; }
  void SetmyNPEMloose(int param) { myZeeStuff.outNPEMloose=param; }
  void SetmyNele(int param)      { myZeeStuff.outNele=param; }     
  void SetmyNeleL(int param)     { myZeeStuff.outNeleL=param; }     
  void SetmyNeleE(int param)     { myZeeStuff.outNeleE=param; }     
  void SetmyZee(TLorentzVector vec)    { myZeeStuff.outMyZee=vec; }
  void SetmyZeeTrk(TLorentzVector vec) { myZeeStuff.outMyZeeTrk=vec; }
  void SetmyZeeFid(int param)    { myZeeStuff.outZeeFid=param; }                     
  void SetmyZeeStat(int param)   { myZeeStuff.outZeeStat=param; }                       
  void SetmyZeeQual(int param)   { myZeeStuff.outZeeQual=param; } 
  void SetmyZeeCharge(int param) { myZeeStuff.outZeeCharge=param; } 

  void SetmyZeeTrkInd(int i, int param) 
  { 
    if(i>=0 && i<myZeeStuff.outTrkInd.size()) myZeeStuff.outTrkInd[i]=param; 
    else myZeeStuff.outTrkInd.push_back(param);
  }
  void SetmyZeeEleInd(int i, int param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleInd.size()) myZeeStuff.outEleInd[i]=param; 
    else myZeeStuff.outEleInd.push_back(param);
  }
  void SetmyEleFid(int i, int param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleFid.size()) myZeeStuff.outEleFid[i]=param; 
    else myZeeStuff.outEleFid.push_back(param);
  }
  void SetmyEleStat(int i, int param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleStat.size()) myZeeStuff.outEleStat[i]=param; 
    else myZeeStuff.outEleStat.push_back(param);
  }
  void SetmyEleCharge(int i, int param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleCharge.size()) myZeeStuff.outEleCharge[i]=param; 
    else myZeeStuff.outEleCharge.push_back(param);
  }
  void SetmyEleConvStat(int i, int param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleConvStat.size()) myZeeStuff.outEleConvStat[i]=param; 
    else myZeeStuff.outEleConvStat.push_back(param);
  }
  void SetmyEleIso(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleIso.size()) myZeeStuff.outEleIso[i]=param; 
    else myZeeStuff.outEleIso.push_back(param);
  }
  void SetmyEleIsoRaw(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleIsoRaw.size()) myZeeStuff.outEleIsoRaw[i]=param; 
    else myZeeStuff.outEleIsoRaw.push_back(param);
  }
  void SetmyEleChi2(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleChi2.size()) myZeeStuff.outEleChi2[i]=param; 
    else myZeeStuff.outEleChi2.push_back(param);
  }
  void SetmyEleHadEm(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleHadEm.size()) myZeeStuff.outEleHadEm[i]=param; 
    else myZeeStuff.outEleHadEm.push_back(param);
  }
  void SetmyEleDetEta(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleDetEta.size()) myZeeStuff.outEleDetEta[i]=param; 
    else myZeeStuff.outEleDetEta.push_back(param);
  }
  void SetmyEleClusPhi(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleClusPhi.size()) myZeeStuff.outEleClusPhi[i]=param; 
    else myZeeStuff.outEleClusPhi.push_back(param);
  }
  void SetmyEleCprPpr(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleCprPpr.size()) myZeeStuff.outEleCprPpr[i]=param; 
    else myZeeStuff.outEleCprPpr.push_back(param);
  }
  void SetmyEleXces(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleXces.size()) myZeeStuff.outEleXces[i]=param; 
    else myZeeStuff.outEleXces.push_back(param);
  }
  void SetmyEleZces(int i, double param) 
  { 
    if(i>=0 && i<myZeeStuff.outEleZces.size()) myZeeStuff.outEleZces[i]=param; 
    else myZeeStuff.outEleZces.push_back(param);
  }
  void SetmyEle(int i, TLorentzVector vec) 
  { 
    if(i>=0 && i<myZeeStuff.outMyEle.size()) myZeeStuff.outMyEle[i]=vec; 
    else myZeeStuff.outMyEle.push_back(vec);
  }
  void SetmyEleRaw(int i, TLorentzVector vec) 
  { 
    if(i>=0 && i<myZeeStuff.outMyEleRaw.size()) myZeeStuff.outMyEleRaw[i]=vec; 
    else myZeeStuff.outMyEleRaw.push_back(vec);
  }
  void SetmyEleTrk(int i, TLorentzVector vec) 
  { 
    if(i>=0 && i<myZeeStuff.outMyEleTrk.size()) myZeeStuff.outMyEleTrk[i]=vec; 
    else myZeeStuff.outMyEleTrk.push_back(vec);
  }

  void SetUseVerbose(int param)  { fUseVerbose=param; }
  void SetDatFileName(char* fname)  { fDatFileName=fname; }

					// ****** overloaded methods of 
					// TStnModule
  int BeginJob();
  int BeginRun();
  int Event   (int ientry);
  int EndJob  ();
  
                                 // ****** other methods

  void ClearModuleOutput(); // clears the Module output in the beginning of every event
  void FillModuleOutput(TStnElectron* Ele, TStnTrack* Trk, TStnEvent* Event,int EleInd, int TrkInd, 
			int EleCharge, int EleStat, int EleConvStat, int N_ele, int N_eleL, int N_eleE, 
			int N_cemT, int N_cemL, int N_pemT, int N_pemL);  // fills the Module output 
                                                              // at the end of every event
  void PrintModuleOutput(); // prints the Module output in the end of every event

  void BookHistograms();

  void BookEleGeneralHistograms(ElectronGeneral_t& Hist, const char* Folder);
  void BookCEMHistograms(CEMElectron_t& Hist, const char* Folder); 
  void BookPEMHistograms(PEMElectron_t& Hist, const char* Folder); 
  void BookZeeHistograms(ZeeGeneral_t& Hist, const char* Folder); 
  void BookConversionHistograms(ConversionHisto_t& Hist, const char* Folder);

  void FillEleGeneralHistogramsB(ElectronGeneral_t& Hist, TStnElectron* Ele);
  void FillCEMHistogramsB(CEMElectron_t& Hist, TStnElectron* Ele, TStnTrack* Trk, int eleInd, int convStat); 
  void FillPEMHistogramsB(PEMElectron_t& Hist, TStnEvent* Event, 
			  TStnElectron* Ele, TStnTrack* Trk, int eleInd, int convStat); 
  void FillZeeHistogramsB(ZeeGeneral_t& Hist, ZeeOutput stuffout); 

  void FillEleGeneralHistogramsA(ElectronGeneral_t& Hist, TStnElectron* Ele);
  void FillCEMHistogramsA(CEMElectron_t& Hist, TStnElectron* Ele, TStnTrack* Trk, int eleInd, int convStat); 
  void FillPEMHistogramsA(PEMElectron_t& Hist, TStnEvent* Event, 
			  TStnElectron* Ele, TStnTrack* Trk, int eleInd, int convStat); 
  void FillZeeHistogramsA(ZeeGeneral_t& Hist, ZeeOutput stuffout); 
  void FillConversionHistograms(ConversionHisto_t& Hist, int eleInd, int triStat);

  char* HistoSummary(TH1F* myhisto);

  //------- cuts
  int MyCemLooseIDcut(TStnElectron* Ele, TStnTrack* Trk, int eleInd, int ConvStat);
  int MyCemTightIDcut(TStnElectron* Ele, TStnTrack* Trk, int eleInd);
  int MyExtraCemCut(TStnElectron* Ele, TStnTrack* Trk); // extra cut to remove events with badly measured electrons
  int MyPemLooseIDcut(TStnElectron* Ele, int eleInd);
  int MyPemTightIDcut(TStnElectron* Ele, TStnTrack* Trk, int eleInd);
  int MyZeeIDcut(ZeeOutput stuffout);

  //------- service functions
  double myRawRconv(TStnTrack* Trk);

  TStnTrack* GetMyPnxTrack(TStnEvent* event, TStnElectron* ele); // my routine to get Phoenix track
                                                                   // by using Ray's methods

  void myHelixToCircle(TStnTrack* Trk, double &x, double &y, double &R); // Helix-to-Cirlce
  void myConversionComputer(TStnTrack* Trk1, TStnTrack* Trk2, 
			    double &dXY, double &dCot, double &R, double &convD0, 
			    double &dPhi, TLorentzVector &conv); //calculates conversion parameters
  int myConversionSelector(double dXY, double dCot, double R, 
			   double &convChi2); //applies cuts on conversion parameters
  int myConversionFilter(TStnTrack* Trk, int trkInd, int eleInd, 
			  TStnTrackBlock* fTrackBlck, 
			  TStnElectronBlock* fEleBlck, int& triStatus); //handles conversions
  double myConversionImpact(double x[3], double y[3],  
			    double R[2], double z); // computes impact parameter for conversion photon

  double myCorrIso(TStnElectron* ele); // returns Iso corrected for leakage & nvx12
  double myCorrIsoFr(TStnElectron* ele); // returns Iso/Et(ele) corrected for leakage & nvx12
  double myCorrEtEle(TStnElectron* ele); // returns corrected Et of electrons (includes all corrections)
  double myCorrEnergyEle(TStnElectron* ele); // returns corrected E of electrons (includes all corrections)

  ClassDef(TMyZeeFilterModule,0)

};

#endif
